/*
 * Copyright 2013 Stormpath, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.stormpath.spring.security.provider;

import com.stormpath.sdk.account.Account;
import com.stormpath.spring.security.authz.permission.Permission;

import java.util.Set;

/**
 * An {@code AccountPermissionResolver} inspects a Stormpath {@link Account} and returns that {@code Account}'s
 * directly assigned {@link com.stormpath.spring.security.authz.permission.Permission}s.
 * <p/>
 * Note that this interface is for resolving permissions that are directly assigned to an Account.  Permissions
 * that are assigned to an account's groups (and therefore implicitly associated with an Account), would be resolved
 * instead by a {@link GroupPermissionResolver} instance instead.
 * <p/>
 *
 * @since 0.2.0
 * @see GroupPermissionResolver
 */
public interface AccountPermissionResolver {

    /**
     * Returns a set of {@link com.stormpath.spring.security.authz.permission.Permission}s assigned to a particular Stormpath
     * {@link Account}.
     * <p/>
     * Note that method is for resolving permissions that are directly assigned to an Account.  Permissions
     * that are assigned to an account's groups (and therefore implicitly associated with an Account), would be resolved
     * instead by a {@link GroupPermissionResolver} instance.
     * <p/>
     * @param account the Stormpath {@code Account} to inspect to return its directly assigned permissions.
     * @return a set of {@link com.stormpath.spring.security.authz.permission.Permission Permission}s assigned to the account, to be
     *         used by Spring Security for runtime permission checks.
     * @see GroupPermissionResolver
     */
     Set<Permission> resolvePermissions(Account account);

}
