/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.RateLimiter;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import javax.annotation.Nullable;
import org.apache.cassandra.db.UnknownColumnFamilyException;
import org.apache.cassandra.hints.ChecksummedDataInput;
import org.apache.cassandra.hints.CompressedChecksummedDataInput;
import org.apache.cassandra.hints.EncryptedChecksummedDataInput;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.InputPosition;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HintsReader
implements AutoCloseable,
Iterable<Page> {
    private static final Logger logger = LoggerFactory.getLogger(HintsReader.class);
    private static final int PAGE_SIZE = 524288;
    private final HintsDescriptor descriptor;
    private final File file;
    private final ChecksummedDataInput input;
    @Nullable
    private final RateLimiter rateLimiter;

    protected HintsReader(HintsDescriptor descriptor, File file, ChecksummedDataInput reader, RateLimiter rateLimiter) {
        this.descriptor = descriptor;
        this.file = file;
        this.input = reader;
        this.rateLimiter = rateLimiter;
    }

    static HintsReader open(File file, RateLimiter rateLimiter) {
        ChecksummedDataInput reader = ChecksummedDataInput.open(file);
        try {
            HintsDescriptor descriptor = HintsDescriptor.deserialize(reader);
            if (descriptor.isCompressed()) {
                reader = CompressedChecksummedDataInput.upgradeInput(reader, descriptor.createCompressor());
            } else if (descriptor.isEncrypted()) {
                reader = EncryptedChecksummedDataInput.upgradeInput(reader, descriptor.getCipher(), descriptor.createCompressor());
            }
            return new HintsReader(descriptor, file, reader, rateLimiter);
        }
        catch (IOException e) {
            reader.close();
            throw new FSReadError((Throwable)e, file);
        }
    }

    static HintsReader open(File file) {
        return HintsReader.open(file, null);
    }

    @Override
    public void close() {
        this.input.close();
    }

    public HintsDescriptor descriptor() {
        return this.descriptor;
    }

    void seek(InputPosition newPosition) {
        this.input.seek(newPosition);
    }

    @Override
    public Iterator<Page> iterator() {
        return new PagesIterator();
    }

    public ChecksummedDataInput getInput() {
        return this.input;
    }

    final class BuffersIterator
    extends AbstractIterator<ByteBuffer> {
        private final InputPosition offset;

        BuffersIterator(InputPosition offset) {
            this.offset = offset;
        }

        @Override
        protected ByteBuffer computeNext() {
            ByteBuffer buffer;
            do {
                InputPosition position = HintsReader.this.input.getSeekPosition();
                if (HintsReader.this.input.isEOF()) {
                    return (ByteBuffer)this.endOfData();
                }
                if (position.subtract(this.offset) >= 524288L) {
                    return (ByteBuffer)this.endOfData();
                }
                try {
                    buffer = this.computeNextInternal();
                }
                catch (EOFException e) {
                    logger.warn("Unexpected EOF replaying hints ({}), likely due to unflushed hint file on shutdown; continuing", (Object)HintsReader.this.descriptor.fileName(), (Object)e);
                    return (ByteBuffer)this.endOfData();
                }
                catch (IOException e) {
                    throw new FSReadError((Throwable)e, HintsReader.this.file);
                }
            } while (buffer == null);
            return buffer;
        }

        private ByteBuffer computeNextInternal() throws IOException {
            HintsReader.this.input.resetCrc();
            HintsReader.this.input.resetLimit();
            int size = HintsReader.this.input.readInt();
            if (!HintsReader.this.input.checkCrc()) {
                throw new IOException("Digest mismatch exception");
            }
            return this.readBuffer(size);
        }

        private ByteBuffer readBuffer(int size) throws IOException {
            if (HintsReader.this.rateLimiter != null) {
                HintsReader.this.rateLimiter.acquire(size);
            }
            HintsReader.this.input.limit(size);
            ByteBuffer buffer = ByteBufferUtil.read(HintsReader.this.input, size);
            if (HintsReader.this.input.checkCrc()) {
                return buffer;
            }
            logger.warn("Failed to read a hint for {} - digest mismatch for hint at position {} in file {}", new Object[]{((HintsReader)HintsReader.this).descriptor.hostId, HintsReader.this.input.getPosition() - (long)size - 4L, HintsReader.this.descriptor.fileName()});
            return null;
        }
    }

    final class HintsIterator
    extends AbstractIterator<Hint> {
        private final InputPosition offset;

        HintsIterator(InputPosition offset) {
            this.offset = offset;
        }

        @Override
        protected Hint computeNext() {
            Hint hint;
            do {
                InputPosition position = HintsReader.this.input.getSeekPosition();
                if (HintsReader.this.input.isEOF()) {
                    return (Hint)this.endOfData();
                }
                if (position.subtract(this.offset) >= 524288L) {
                    return (Hint)this.endOfData();
                }
                try {
                    hint = this.computeNextInternal();
                }
                catch (EOFException e) {
                    logger.warn("Unexpected EOF replaying hints ({}), likely due to unflushed hint file on shutdown; continuing", (Object)HintsReader.this.descriptor.fileName(), (Object)e);
                    return (Hint)this.endOfData();
                }
                catch (IOException e) {
                    throw new FSReadError((Throwable)e, HintsReader.this.file);
                }
            } while (hint == null);
            return hint;
        }

        private Hint computeNextInternal() throws IOException {
            HintsReader.this.input.resetCrc();
            HintsReader.this.input.resetLimit();
            int size = HintsReader.this.input.readInt();
            if (!HintsReader.this.input.checkCrc()) {
                throw new IOException("Digest mismatch exception");
            }
            return this.readHint(size);
        }

        private Hint readHint(int size) throws IOException {
            Hint hint;
            if (HintsReader.this.rateLimiter != null) {
                HintsReader.this.rateLimiter.acquire(size);
            }
            HintsReader.this.input.limit(size);
            try {
                hint = Hint.serializer.deserialize(HintsReader.this.input, HintsReader.this.descriptor.messagingVersion());
                HintsReader.this.input.checkLimit(0);
            }
            catch (UnknownColumnFamilyException e) {
                logger.warn("Failed to read a hint for {}: {} - table with id {} is unknown in file {}", new Object[]{StorageService.instance.getEndpointForHostId(((HintsReader)HintsReader.this).descriptor.hostId), ((HintsReader)HintsReader.this).descriptor.hostId, e.cfId, HintsReader.this.descriptor.fileName()});
                HintsReader.this.input.skipBytes(Ints.checkedCast((long)((long)size - HintsReader.this.input.bytesPastLimit())));
                return null;
            }
            if (HintsReader.this.input.checkCrc()) {
                return hint;
            }
            logger.warn("Failed to read a hint for {}: {} - digest mismatch for hint at position {} in file {}", new Object[]{StorageService.instance.getEndpointForHostId(((HintsReader)HintsReader.this).descriptor.hostId), ((HintsReader)HintsReader.this).descriptor.hostId, HintsReader.this.input.getPosition() - (long)size - 4L, HintsReader.this.descriptor.fileName()});
            return null;
        }
    }

    final class PagesIterator
    extends AbstractIterator<Page> {
        PagesIterator() {
        }

        @Override
        protected Page computeNext() {
            HintsReader.this.input.tryUncacheRead();
            if (HintsReader.this.input.isEOF()) {
                return (Page)this.endOfData();
            }
            return new Page(HintsReader.this.input.getSeekPosition());
        }
    }

    final class Page {
        public final InputPosition position;

        private Page(InputPosition inputPosition) {
            this.position = inputPosition;
        }

        Iterator<Hint> hintsIterator() {
            return new HintsIterator(this.position);
        }

        Iterator<ByteBuffer> buffersIterator() {
            return new BuffersIterator(this.position);
        }
    }
}

