/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.codahale.metrics.Timer;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.CRC32;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CompressedSegment;
import org.apache.cassandra.db.commitlog.EncryptedSegment;
import org.apache.cassandra.db.commitlog.MemoryMappedSegment;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.IntegerInterval;
import org.apache.cassandra.utils.NativeLibrary;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.cliffc.high_scale_lib.NonBlockingHashMap;

public abstract class CommitLogSegment {
    private static final long idBase;
    private CDCState cdcState = CDCState.PERMITTED;
    Object cdcStateLock = new Object();
    private static final AtomicInteger nextId;
    private static long replayLimitId;
    public static final int ENTRY_OVERHEAD_SIZE = 12;
    static final int SYNC_MARKER_SIZE = 8;
    private final OpOrder appendOrder = new OpOrder();
    private final AtomicInteger allocatePosition = new AtomicInteger();
    private volatile int lastSyncedOffset;
    private int endOfBuffer;
    private final WaitQueue syncComplete = new WaitQueue();
    private final NonBlockingHashMap<UUID, IntegerInterval> cfDirty = new NonBlockingHashMap(1024);
    private final ConcurrentHashMap<UUID, IntegerInterval.Set> cfClean = new ConcurrentHashMap();
    public final long id;
    final File logFile;
    final FileChannel channel;
    final int fd;
    protected final AbstractCommitLogSegmentManager manager;
    ByteBuffer buffer;
    private volatile boolean headerWritten;
    public final CommitLogDescriptor descriptor;

    static CommitLogSegment createSegment(CommitLog commitLog, AbstractCommitLogSegmentManager manager) {
        CommitLog.Configuration config = commitLog.configuration;
        CommitLogSegment segment = config.useEncryption() ? new EncryptedSegment(commitLog, manager) : (config.useCompression() ? new CompressedSegment(commitLog, manager) : new MemoryMappedSegment(commitLog, manager));
        segment.writeLogHeader();
        return segment;
    }

    static boolean usesBufferPool(CommitLog commitLog) {
        CommitLog.Configuration config = commitLog.configuration;
        return config.useEncryption() || config.useCompression();
    }

    static long getNextId() {
        return idBase + (long)nextId.getAndIncrement();
    }

    CommitLogSegment(CommitLog commitLog, AbstractCommitLogSegmentManager manager) {
        this.manager = manager;
        this.id = CommitLogSegment.getNextId();
        this.descriptor = new CommitLogDescriptor(this.id, commitLog.configuration.getCompressorClass(), commitLog.configuration.getEncryptionContext());
        this.logFile = new File(manager.storageDirectory, this.descriptor.fileName());
        try {
            this.channel = FileChannel.open(this.logFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE);
            this.fd = NativeLibrary.getfd(this.channel);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.logFile);
        }
        this.buffer = this.createBuffer(commitLog);
    }

    void writeLogHeader() {
        CommitLogDescriptor.writeHeader(this.buffer, this.descriptor, this.additionalHeaderParameters());
        this.endOfBuffer = this.buffer.capacity();
        this.lastSyncedOffset = this.buffer.position();
        this.allocatePosition.set(this.lastSyncedOffset + 8);
        this.headerWritten = true;
    }

    protected Map<String, String> additionalHeaderParameters() {
        return Collections.emptyMap();
    }

    abstract ByteBuffer createBuffer(CommitLog var1);

    Allocation allocate(Mutation mutation, int size) {
        OpOrder.Group opGroup = this.appendOrder.start();
        try {
            int position = this.allocate(size);
            if (position < 0) {
                opGroup.close();
                return null;
            }
            this.markDirty(mutation, position);
            return new Allocation(this, opGroup, position, (ByteBuffer)this.buffer.duplicate().position(position).limit(position + size));
        }
        catch (Throwable t) {
            opGroup.close();
            throw t;
        }
    }

    static boolean shouldReplay(String name) {
        return CommitLogDescriptor.fromFileName((String)name).id < replayLimitId;
    }

    static void resetReplayLimit() {
        replayLimitId = CommitLogSegment.getNextId();
    }

    private int allocate(int size) {
        int next;
        int prev;
        do {
            if ((next = (prev = this.allocatePosition.get()) + size) < this.endOfBuffer) continue;
            return -1;
        } while (!this.allocatePosition.compareAndSet(prev, next));
        assert (this.buffer != null);
        return prev;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    void discardUnusedTail() {
        try (OpOrder.Group group = this.appendOrder.start();){
            while (true) {
                int next;
                int prev;
                if ((prev = this.allocatePosition.get()) >= (next = this.endOfBuffer + 1)) {
                    assert (this.buffer == null || prev == this.buffer.capacity() + 1);
                    return;
                }
                if (this.allocatePosition.compareAndSet(prev, next)) {
                    this.endOfBuffer = prev;
                    assert (this.buffer != null && next == this.buffer.capacity() + 1);
                    return;
                }
                continue;
                break;
            }
        }
    }

    void waitForModifications() {
        this.appendOrder.awaitNewBarrier();
    }

    synchronized void sync() {
        if (!this.headerWritten) {
            throw new IllegalStateException("commit log header has not been written");
        }
        boolean close = false;
        if (this.allocatePosition.get() <= this.lastSyncedOffset + 8) {
            return;
        }
        assert (this.buffer != null);
        int startMarker = this.lastSyncedOffset;
        int nextMarker = this.allocate(8);
        if (nextMarker < 0) {
            this.discardUnusedTail();
            close = true;
            nextMarker = this.buffer.capacity();
        }
        this.waitForModifications();
        int sectionEnd = close ? this.endOfBuffer : nextMarker;
        this.write(startMarker, sectionEnd);
        this.lastSyncedOffset = nextMarker;
        if (close) {
            this.internalClose();
        }
        this.syncComplete.signalAll();
    }

    protected void writeSyncMarker(ByteBuffer buffer, int offset, int filePos, int nextMarker) {
        if (filePos > nextMarker) {
            throw new IllegalArgumentException(String.format("commit log sync marker's current file position %d is greater than next file position %d", filePos, nextMarker));
        }
        CRC32 crc = new CRC32();
        FBUtilities.updateChecksumInt(crc, (int)(this.id & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(this.id >>> 32));
        FBUtilities.updateChecksumInt(crc, filePos);
        buffer.putInt(offset, nextMarker);
        buffer.putInt(offset + 4, (int)crc.getValue());
    }

    abstract void write(int var1, int var2);

    public boolean isStillAllocating() {
        return this.allocatePosition.get() < this.endOfBuffer;
    }

    void discard(boolean deleteFile) {
        this.close();
        if (deleteFile) {
            FileUtils.deleteWithConfirm(this.logFile);
        }
        this.manager.addSize(-this.onDiskSize());
    }

    public CommitLogPosition getCurrentCommitLogPosition() {
        return new CommitLogPosition(this.id, this.allocatePosition.get());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    void waitForFinalSync() {
        WaitQueue.Signal signal;
        while (true) {
            signal = this.syncComplete.register();
            if (this.lastSyncedOffset >= this.endOfBuffer) break;
            signal.awaitUninterruptibly();
        }
        signal.cancel();
    }

    void waitForSync(int position, Timer waitingOnCommit) {
        while (this.lastSyncedOffset < position) {
            WaitQueue.Signal signal;
            WaitQueue.Signal signal2 = signal = waitingOnCommit != null ? this.syncComplete.register(waitingOnCommit.time()) : this.syncComplete.register();
            if (this.lastSyncedOffset < position) {
                signal.awaitUninterruptibly();
                continue;
            }
            signal.cancel();
        }
    }

    synchronized void close() {
        this.discardUnusedTail();
        this.sync();
        assert (this.buffer == null);
    }

    protected void internalClose() {
        try {
            this.channel.close();
            this.buffer = null;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    public static <K> void coverInMap(ConcurrentMap<K, IntegerInterval> map, K key, int value) {
        IntegerInterval i = (IntegerInterval)map.get(key);
        if (i == null && (i = map.putIfAbsent(key, new IntegerInterval(value, value))) == null) {
            return;
        }
        i.expandToCover(value);
    }

    void markDirty(Mutation mutation, int allocatedPosition) {
        for (PartitionUpdate update : mutation.getPartitionUpdates()) {
            CommitLogSegment.coverInMap(this.cfDirty, update.metadata().cfId, allocatedPosition);
        }
    }

    public synchronized void markClean(UUID cfId, CommitLogPosition startPosition, CommitLogPosition endPosition) {
        if (startPosition.segmentId > this.id || endPosition.segmentId < this.id) {
            return;
        }
        if (!this.cfDirty.containsKey((Object)cfId)) {
            return;
        }
        int start = startPosition.segmentId == this.id ? startPosition.position : 0;
        int end = endPosition.segmentId == this.id ? endPosition.position : Integer.MAX_VALUE;
        this.cfClean.computeIfAbsent(cfId, k -> new IntegerInterval.Set()).add(start, end);
        this.removeCleanFromDirty();
    }

    private void removeCleanFromDirty() {
        if (this.isStillAllocating()) {
            return;
        }
        Iterator<Map.Entry<UUID, IntegerInterval.Set>> iter = this.cfClean.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<UUID, IntegerInterval.Set> clean = iter.next();
            UUID cfId = clean.getKey();
            IntegerInterval.Set cleanSet = clean.getValue();
            IntegerInterval dirtyInterval = (IntegerInterval)this.cfDirty.get((Object)cfId);
            if (dirtyInterval == null || !cleanSet.covers(dirtyInterval)) continue;
            this.cfDirty.remove((Object)cfId);
            iter.remove();
        }
    }

    public synchronized Collection<UUID> getDirtyCFIDs() {
        if (this.cfClean.isEmpty() || this.cfDirty.isEmpty()) {
            return this.cfDirty.keySet();
        }
        ArrayList<UUID> r = new ArrayList<UUID>(this.cfDirty.size());
        for (Map.Entry dirty : this.cfDirty.entrySet()) {
            UUID cfId = (UUID)dirty.getKey();
            IntegerInterval dirtyInterval = (IntegerInterval)dirty.getValue();
            IntegerInterval.Set cleanSet = this.cfClean.get(cfId);
            if (cleanSet != null && cleanSet.covers(dirtyInterval)) continue;
            r.add((UUID)dirty.getKey());
        }
        return r;
    }

    public synchronized boolean isUnused() {
        if (this.isStillAllocating()) {
            return false;
        }
        this.removeCleanFromDirty();
        return this.cfDirty.isEmpty();
    }

    public boolean contains(CommitLogPosition context) {
        return context.segmentId == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (UUID cfId : this.getDirtyCFIDs()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append(", dirty: ").append(this.cfDirty.get((Object)cfId)).append(", clean: ").append(this.cfClean.get(cfId)).append("), ");
        }
        return sb.toString();
    }

    public abstract long onDiskSize();

    public long contentSize() {
        return this.lastSyncedOffset;
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    public CDCState getCDCState() {
        return this.cdcState;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setCDCState(CDCState newState) {
        if (newState == this.cdcState) {
            return;
        }
        Object object = this.cdcStateLock;
        synchronized (object) {
            if (this.cdcState == CDCState.CONTAINS && newState != CDCState.CONTAINS) {
                throw new IllegalArgumentException("Cannot transition from CONTAINS to any other state.");
            }
            if (this.cdcState == CDCState.FORBIDDEN && newState != CDCState.PERMITTED) {
                throw new IllegalArgumentException("Only transition from FORBIDDEN to PERMITTED is allowed.");
            }
            this.cdcState = newState;
        }
    }

    static {
        nextId = new AtomicInteger(1);
        long maxId = Long.MIN_VALUE;
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles()) {
            if (!CommitLogDescriptor.isValid(file.getName())) continue;
            maxId = Math.max(CommitLogDescriptor.fromFileName((String)file.getName()).id, maxId);
        }
        replayLimitId = idBase = Math.max(System.currentTimeMillis(), maxId + 1L);
    }

    protected static class Allocation {
        private final CommitLogSegment segment;
        private final OpOrder.Group appendOp;
        private final int position;
        private final ByteBuffer buffer;

        Allocation(CommitLogSegment segment, OpOrder.Group appendOp, int position, ByteBuffer buffer) {
            this.segment = segment;
            this.appendOp = appendOp;
            this.position = position;
            this.buffer = buffer;
        }

        CommitLogSegment getSegment() {
            return this.segment;
        }

        ByteBuffer getBuffer() {
            return this.buffer;
        }

        void markWritten() {
            this.appendOp.close();
        }

        void awaitDiskSync(Timer waitingOnCommit) {
            this.segment.waitForSync(this.position, waitingOnCommit);
        }

        public CommitLogPosition getCommitLogPosition() {
            return new CommitLogPosition(this.segment.id, this.buffer.limit());
        }
    }

    public static class CommitLogSegmentFileComparator
    implements Comparator<File> {
        @Override
        public int compare(File f, File f2) {
            CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(f.getName());
            CommitLogDescriptor desc2 = CommitLogDescriptor.fromFileName(f2.getName());
            return Long.compare(desc.id, desc2.id);
        }
    }

    public static enum CDCState {
        PERMITTED,
        FORBIDDEN,
        CONTAINS;

    }
}

