/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.base.Preconditions;
import com.google.common.collect.PeekingIterator;
import java.io.DataInput;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.IPartitionerDependentSerializer;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.RingPosition;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Hex;
import org.apache.cassandra.utils.HistogramBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MerkleTree
implements Serializable {
    private static Logger logger = LoggerFactory.getLogger(MerkleTree.class);
    public static final MerkleTreeSerializer serializer = new MerkleTreeSerializer();
    private static final long serialVersionUID = 2L;
    public static final byte RECOMMENDED_DEPTH = 126;
    public static final int CONSISTENT = 0;
    public static final int FULLY_INCONSISTENT = 1;
    public static final int PARTIALLY_INCONSISTENT = 2;
    private static final byte[] EMPTY_HASH = new byte[0];
    public final byte hashdepth;
    public final Range<Token> fullRange;
    private final IPartitioner partitioner;
    private long maxsize;
    private long size;
    private Hashable root;

    public MerkleTree(IPartitioner partitioner, Range<Token> range, byte hashdepth, long maxsize) {
        assert (hashdepth < 127);
        this.fullRange = (Range)Preconditions.checkNotNull(range);
        this.partitioner = (IPartitioner)Preconditions.checkNotNull((Object)partitioner);
        this.hashdepth = hashdepth;
        this.maxsize = maxsize;
        this.size = 1L;
        this.root = new Leaf(null);
    }

    static byte inc(byte in) {
        assert (in < 127);
        return (byte)(in + 1);
    }

    public void init() {
        byte sizedepth = (byte)(Math.log10(this.maxsize) / Math.log10(2.0));
        byte depth = (byte)Math.min(sizedepth, this.hashdepth);
        this.root = this.initHelper((Token)this.fullRange.left, (Token)this.fullRange.right, (byte)0, depth);
        this.size = (long)Math.pow(2.0, depth);
    }

    private Hashable initHelper(Token left, Token right, byte depth, byte max) {
        if (depth == max) {
            return new Leaf();
        }
        Token midpoint = this.partitioner.midpoint(left, right);
        if (midpoint.equals(left) || midpoint.equals(right)) {
            return new Leaf();
        }
        Hashable lchild = this.initHelper(left, midpoint, MerkleTree.inc(depth), max);
        Hashable rchild = this.initHelper(midpoint, right, MerkleTree.inc(depth), max);
        return new Inner(midpoint, lchild, rchild);
    }

    Hashable root() {
        return this.root;
    }

    public IPartitioner partitioner() {
        return this.partitioner;
    }

    public long size() {
        return this.size;
    }

    public long maxsize() {
        return this.maxsize;
    }

    public void maxsize(long maxsize) {
        this.maxsize = maxsize;
    }

    public static List<TreeRange> difference(MerkleTree ltree, MerkleTree rtree) {
        if (!ltree.fullRange.equals(rtree.fullRange)) {
            throw new IllegalArgumentException("Difference only make sense on tree covering the same range (but " + ltree.fullRange + " != " + rtree.fullRange + ")");
        }
        ArrayList<TreeRange> diff = new ArrayList<TreeRange>();
        TreeDifference active = new TreeDifference((Token)ltree.fullRange.left, (Token)ltree.fullRange.right, 0);
        Hashable lnode = ltree.find(active);
        Hashable rnode = rtree.find(active);
        byte[] lhash = lnode.hash();
        byte[] rhash = rnode.hash();
        active.setSize(lnode.sizeOfRange(), rnode.sizeOfRange());
        if (lhash != null && rhash != null && !Arrays.equals(lhash, rhash)) {
            logger.debug("Digest mismatch detected, traversing trees [{}, {}]", (Object)ltree, (Object)rtree);
            if (1 == MerkleTree.differenceHelper(ltree, rtree, diff, active)) {
                logger.debug("Range {} fully inconsistent", (Object)active);
                diff.add(active);
            }
        } else if (lhash == null || rhash == null) {
            diff.add(active);
        }
        return diff;
    }

    static int differenceHelper(MerkleTree ltree, MerkleTree rtree, List<TreeRange> diff, TreeRange active) {
        boolean rreso;
        boolean lreso;
        if (active.depth == 127) {
            return 0;
        }
        Token midpoint = ltree.partitioner().midpoint((Token)active.left, (Token)active.right);
        if (midpoint.equals(active.left) || midpoint.equals(active.right)) {
            logger.error("Invalid midpoint {} for [{},{}], range will be reported inconsistent", new Object[]{midpoint, active.left, active.right});
            return 1;
        }
        TreeDifference left = new TreeDifference((Token)active.left, midpoint, MerkleTree.inc(active.depth));
        TreeDifference right = new TreeDifference(midpoint, (Token)active.right, MerkleTree.inc(active.depth));
        logger.debug("({}) Hashing sub-ranges [{}, {}] for {} divided by midpoint {}", new Object[]{active.depth, left, right, active, midpoint});
        Hashable lnode = ltree.find(left);
        Hashable rnode = rtree.find(left);
        byte[] lhash = lnode.hash();
        byte[] rhash = rnode.hash();
        left.setSize(lnode.sizeOfRange(), rnode.sizeOfRange());
        left.setRows(lnode.rowsInRange(), rnode.rowsInRange());
        int ldiff = 0;
        boolean bl = lreso = lhash != null && rhash != null;
        if (lreso && !Arrays.equals(lhash, rhash)) {
            logger.debug("({}) Inconsistent digest on left sub-range {}: [{}, {}]", new Object[]{active.depth, left, lnode, rnode});
            ldiff = lnode instanceof Leaf ? 1 : MerkleTree.differenceHelper(ltree, rtree, diff, left);
        } else if (!lreso) {
            logger.debug("({}) Left sub-range fully inconsistent {}", (Object)active.depth, (Object)right);
            ldiff = 1;
        }
        lnode = ltree.find(right);
        rnode = rtree.find(right);
        lhash = lnode.hash();
        rhash = rnode.hash();
        right.setSize(lnode.sizeOfRange(), rnode.sizeOfRange());
        right.setRows(lnode.rowsInRange(), rnode.rowsInRange());
        int rdiff = 0;
        boolean bl2 = rreso = lhash != null && rhash != null;
        if (rreso && !Arrays.equals(lhash, rhash)) {
            logger.debug("({}) Inconsistent digest on right sub-range {}: [{}, {}]", new Object[]{active.depth, right, lnode, rnode});
            rdiff = rnode instanceof Leaf ? 1 : MerkleTree.differenceHelper(ltree, rtree, diff, right);
        } else if (!rreso) {
            logger.debug("({}) Right sub-range fully inconsistent {}", (Object)active.depth, (Object)right);
            rdiff = 1;
        }
        if (ldiff == 1 && rdiff == 1) {
            logger.debug("({}) Fully inconsistent range [{}, {}]", new Object[]{active.depth, left, right});
            return 1;
        }
        if (ldiff == 1) {
            logger.debug("({}) Adding left sub-range to diff as fully inconsistent {}", (Object)active.depth, (Object)left);
            diff.add(left);
            return 2;
        }
        if (rdiff == 1) {
            logger.debug("({}) Adding right sub-range to diff as fully inconsistent {}", (Object)active.depth, (Object)right);
            diff.add(right);
            return 2;
        }
        logger.debug("({}) Range {} partially inconstent", (Object)active.depth, (Object)active);
        return 2;
    }

    public TreeRange get(Token t) {
        return this.getHelper(this.root, (Token)this.fullRange.left, (Token)this.fullRange.right, (byte)0, t);
    }

    TreeRange getHelper(Hashable hashable, Token pleft, Token pright, byte depth, Token t) {
        while (!(hashable instanceof Leaf)) {
            Inner node = (Inner)hashable;
            depth = MerkleTree.inc(depth);
            if (Range.contains(pleft, node.token, t)) {
                hashable = node.lchild;
                pright = node.token;
                continue;
            }
            hashable = node.rchild;
            pleft = node.token;
        }
        return new TreeRange(this, pleft, pright, depth, hashable);
    }

    public void invalidate(Token t) {
        this.invalidateHelper(this.root, (Token)this.fullRange.left, t);
    }

    private void invalidateHelper(Hashable hashable, Token pleft, Token t) {
        hashable.hash(null);
        if (hashable instanceof Leaf) {
            return;
        }
        Inner node = (Inner)hashable;
        if (Range.contains(pleft, node.token, t)) {
            this.invalidateHelper(node.lchild, pleft, t);
        } else {
            this.invalidateHelper(node.rchild, node.token, t);
        }
    }

    public byte[] hash(Range<Token> range) {
        return this.find(range).hash();
    }

    private Hashable find(Range<Token> range) {
        try {
            return this.findHelper(this.root, new Range<RingPosition>(this.fullRange.left, this.fullRange.right), range);
        }
        catch (StopRecursion e) {
            return new Leaf();
        }
    }

    private Hashable findHelper(Hashable current, Range<Token> activeRange, Range<Token> find) throws StopRecursion {
        while (true) {
            if (current instanceof Leaf) {
                if (!find.contains((Token)((Object)activeRange))) {
                    throw new StopRecursion.BadRange();
                }
                return current;
            }
            Inner node = (Inner)current;
            Range<Token> leftRange = new Range<Token>((Token)activeRange.left, node.token);
            Range<RingPosition> rightRange = new Range<RingPosition>(node.token, activeRange.right);
            if (find.contains((Token)((Object)activeRange))) {
                return node.calc();
            }
            if (leftRange.contains((Token)((Object)find))) {
                current = node.lchild;
                activeRange = leftRange;
                continue;
            }
            if (!rightRange.contains((RingPosition)((Object)find))) break;
            current = node.rchild;
            activeRange = rightRange;
        }
        throw new StopRecursion.BadRange();
    }

    public boolean split(Token t) {
        if (this.size >= this.maxsize) {
            return false;
        }
        try {
            this.root = this.splitHelper(this.root, (Token)this.fullRange.left, (Token)this.fullRange.right, (byte)0, t);
        }
        catch (StopRecursion.TooDeep e) {
            return false;
        }
        return true;
    }

    private Hashable splitHelper(Hashable hashable, Token pleft, Token pright, byte depth, Token t) throws StopRecursion.TooDeep {
        if (depth >= this.hashdepth) {
            throw new StopRecursion.TooDeep();
        }
        if (hashable instanceof Leaf) {
            Token midpoint = this.partitioner.midpoint(pleft, pright);
            if (midpoint.equals(pleft) || midpoint.equals(pright)) {
                throw new StopRecursion.TooDeep();
            }
            ++this.size;
            return new Inner(midpoint, new Leaf(), new Leaf());
        }
        Inner node = (Inner)hashable;
        if (Range.contains(pleft, node.token, t)) {
            node.lchild(this.splitHelper(node.lchild, pleft, node.token, MerkleTree.inc(depth), t));
        } else {
            node.rchild(this.splitHelper(node.rchild, node.token, pright, MerkleTree.inc(depth), t));
        }
        return node;
    }

    public TreeRangeIterator invalids() {
        return new TreeRangeIterator(this);
    }

    public EstimatedHistogram histogramOfRowSizePerLeaf() {
        HistogramBuilder histbuild = new HistogramBuilder();
        for (TreeRange range : new TreeRangeIterator(this)) {
            histbuild.add(((TreeRange)range).hashable.sizeOfRange);
        }
        return histbuild.buildWithStdevRangesAroundMean();
    }

    public EstimatedHistogram histogramOfRowCountPerLeaf() {
        HistogramBuilder histbuild = new HistogramBuilder();
        for (TreeRange range : new TreeRangeIterator(this)) {
            histbuild.add(((TreeRange)range).hashable.rowsInRange);
        }
        return histbuild.buildWithStdevRangesAroundMean();
    }

    public long rowCount() {
        long count = 0L;
        for (TreeRange range : new TreeRangeIterator(this)) {
            count += ((TreeRange)range).hashable.rowsInRange;
        }
        return count;
    }

    public String toString() {
        StringBuilder buff = new StringBuilder();
        buff.append("#<MerkleTree root=");
        this.root.toString(buff, 8);
        buff.append(">");
        return buff.toString();
    }

    static abstract class StopRecursion
    extends Exception {
        StopRecursion() {
        }

        static class TooDeep
        extends StopRecursion {
        }

        static class InvalidHash
        extends StopRecursion {
        }

        static class BadRange
        extends StopRecursion {
        }
    }

    static abstract class Hashable
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private static final IPartitionerDependentSerializer<Hashable> serializer = new HashableSerializer();
        protected byte[] hash;
        protected long sizeOfRange;
        protected long rowsInRange;

        protected Hashable(byte[] hash) {
            this.hash = hash;
        }

        public byte[] hash() {
            return this.hash;
        }

        public long sizeOfRange() {
            return this.sizeOfRange;
        }

        public long rowsInRange() {
            return this.rowsInRange;
        }

        void hash(byte[] hash) {
            this.hash = hash;
        }

        Hashable calc() {
            return this;
        }

        void hash(byte[] lefthash, byte[] righthash) {
            this.hash = Hashable.binaryHash(lefthash, righthash);
        }

        void addHash(byte[] righthash, long sizeOfRow) {
            this.hash = this.hash == null ? righthash : Hashable.binaryHash(this.hash, righthash);
            this.sizeOfRange += sizeOfRow;
            ++this.rowsInRange;
        }

        static byte[] binaryHash(byte[] left, byte[] right) {
            return FBUtilities.xor(left, right);
        }

        public abstract void toString(StringBuilder var1, int var2);

        public static String toString(byte[] hash) {
            if (hash == null) {
                return "null";
            }
            return "[" + Hex.bytesToHex(hash) + "]";
        }

        private static class HashableSerializer
        implements IPartitionerDependentSerializer<Hashable> {
            private HashableSerializer() {
            }

            @Override
            public void serialize(Hashable h, DataOutputPlus out, int version) throws IOException {
                if (h instanceof Inner) {
                    out.writeByte(2);
                    Inner.serializer.serialize((Inner)h, out, version);
                } else if (h instanceof Leaf) {
                    out.writeByte(1);
                    Leaf.serializer.serialize((Leaf)h, out, version);
                } else {
                    throw new IOException("Unexpected Hashable: " + h.getClass().getCanonicalName());
                }
            }

            @Override
            public Hashable deserialize(DataInput in, IPartitioner p, int version) throws IOException {
                byte ident = in.readByte();
                if (2 == ident) {
                    return Inner.serializer.deserialize(in, p, version);
                }
                if (1 == ident) {
                    return Leaf.serializer.deserialize(in, p, version);
                }
                throw new IOException("Unexpected Hashable: " + ident);
            }

            @Override
            public long serializedSize(Hashable h, int version) {
                if (h instanceof Inner) {
                    return 1L + Inner.serializer.serializedSize((Inner)h, version);
                }
                if (h instanceof Leaf) {
                    return 1L + Leaf.serializer.serializedSize((Leaf)h, version);
                }
                throw new AssertionError(h.getClass());
            }
        }
    }

    public static class RowHash {
        public final Token token;
        public final byte[] hash;
        public final long size;

        public RowHash(Token token, byte[] hash, long size) {
            this.token = token;
            this.hash = hash;
            this.size = size;
        }

        public String toString() {
            return "#<RowHash " + this.token + " " + Hashable.toString(this.hash) + " @ " + this.size + " bytes>";
        }
    }

    static class Leaf
    extends Hashable {
        public static final long serialVersionUID = 1L;
        static final byte IDENT = 1;
        private static final LeafSerializer serializer = new LeafSerializer();

        public Leaf() {
            super(null);
        }

        public Leaf(byte[] hash) {
            super(hash);
        }

        @Override
        public void toString(StringBuilder buff, int maxdepth) {
            buff.append(this.toString());
        }

        public String toString() {
            return "#<Leaf " + Hashable.toString(this.hash()) + ">";
        }

        private static class LeafSerializer
        implements IPartitionerDependentSerializer<Leaf> {
            private LeafSerializer() {
            }

            @Override
            public void serialize(Leaf leaf, DataOutputPlus out, int version) throws IOException {
                if (leaf.hash == null) {
                    if (version < 10) {
                        out.writeInt(-1);
                    } else {
                        out.writeByte(-1);
                    }
                } else {
                    if (version < 10) {
                        out.writeInt(leaf.hash.length);
                    } else {
                        out.writeByte(leaf.hash.length);
                    }
                    out.write(leaf.hash);
                }
            }

            @Override
            public Leaf deserialize(DataInput in, IPartitioner p, int version) throws IOException {
                byte[] hash;
                int hashLen = version < 10 ? in.readInt() : (int)in.readByte();
                byte[] byArray = hash = hashLen < 0 ? null : new byte[hashLen];
                if (hash != null) {
                    in.readFully(hash);
                }
                return new Leaf(hash);
            }

            @Override
            public long serializedSize(Leaf leaf, int version) {
                long size;
                long l = size = version < 10 ? (long)TypeSizes.sizeof(1) : 1L;
                if (leaf.hash != null) {
                    size += (long)leaf.hash().length;
                }
                return size;
            }
        }
    }

    static class Inner
    extends Hashable {
        public static final long serialVersionUID = 1L;
        static final byte IDENT = 2;
        public final Token token;
        private Hashable lchild;
        private Hashable rchild;
        private static final InnerSerializer serializer = new InnerSerializer();

        public Inner(Token token, Hashable lchild, Hashable rchild) {
            super(null);
            this.token = token;
            this.lchild = lchild;
            this.rchild = rchild;
        }

        public Hashable lchild() {
            return this.lchild;
        }

        public Hashable rchild() {
            return this.rchild;
        }

        public void lchild(Hashable child) {
            this.lchild = child;
        }

        public void rchild(Hashable child) {
            this.rchild = child;
        }

        @Override
        Hashable calc() {
            if (this.hash == null) {
                Hashable lnode = this.lchild.calc();
                Hashable rnode = this.rchild.calc();
                this.hash(lnode.hash, rnode.hash);
                this.sizeOfRange = lnode.sizeOfRange + rnode.sizeOfRange;
                this.rowsInRange = lnode.rowsInRange + rnode.rowsInRange;
            }
            return this;
        }

        @Override
        public void toString(StringBuilder buff, int maxdepth) {
            buff.append("#<").append(this.getClass().getSimpleName());
            buff.append(" ").append(this.token);
            buff.append(" hash=").append(Hashable.toString(this.hash()));
            buff.append(" children=[");
            if (maxdepth < 1) {
                buff.append("#");
            } else {
                if (this.lchild == null) {
                    buff.append("null");
                } else {
                    this.lchild.toString(buff, maxdepth - 1);
                }
                buff.append(" ");
                if (this.rchild == null) {
                    buff.append("null");
                } else {
                    this.rchild.toString(buff, maxdepth - 1);
                }
            }
            buff.append("]>");
        }

        public String toString() {
            StringBuilder buff = new StringBuilder();
            this.toString(buff, 1);
            return buff.toString();
        }

        private static class InnerSerializer
        implements IPartitionerDependentSerializer<Inner> {
            private InnerSerializer() {
            }

            @Override
            public void serialize(Inner inner, DataOutputPlus out, int version) throws IOException {
                if (version < 10) {
                    if (inner.hash == null) {
                        out.writeInt(-1);
                    } else {
                        out.writeInt(inner.hash.length);
                        out.write(inner.hash);
                    }
                }
                Token.serializer.serialize(inner.token, out, version);
                Hashable.serializer.serialize(inner.lchild, out, version);
                Hashable.serializer.serialize(inner.rchild, out, version);
            }

            @Override
            public Inner deserialize(DataInput in, IPartitioner p, int version) throws IOException {
                if (version < 10) {
                    byte[] hash;
                    int hashLen = in.readInt();
                    byte[] byArray = hash = hashLen >= 0 ? new byte[hashLen] : null;
                    if (hash != null) {
                        in.readFully(hash);
                    }
                }
                Token token = Token.serializer.deserialize(in, p, version);
                Hashable lchild = (Hashable)Hashable.serializer.deserialize(in, p, version);
                Hashable rchild = (Hashable)Hashable.serializer.deserialize(in, p, version);
                return new Inner(token, lchild, rchild);
            }

            @Override
            public long serializedSize(Inner inner, int version) {
                long size = 0L;
                if (version < 10) {
                    size += inner.hash == null ? (long)TypeSizes.sizeof(-1) : (long)(TypeSizes.sizeof(inner.hash().length) + inner.hash().length);
                }
                return size += Token.serializer.serializedSize(inner.token, version) + Hashable.serializer.serializedSize(inner.lchild, version) + Hashable.serializer.serializedSize(inner.rchild, version);
            }
        }
    }

    public static class TreeRangeIterator
    extends AbstractIterator<TreeRange>
    implements Iterable<TreeRange>,
    PeekingIterator<TreeRange> {
        private final ArrayDeque<TreeRange> tovisit = new ArrayDeque();
        private final MerkleTree tree;

        TreeRangeIterator(MerkleTree tree) {
            this.tovisit.add(new TreeRange(tree, (Token)tree.fullRange.left, (Token)tree.fullRange.right, 0, tree.root));
            this.tree = tree;
        }

        @Override
        public TreeRange computeNext() {
            while (!this.tovisit.isEmpty()) {
                TreeRange active = this.tovisit.pop();
                if (active.hashable instanceof Leaf) {
                    if (active.isWrapAround() && !this.tovisit.isEmpty()) {
                        this.tovisit.addLast(active);
                    }
                    return active;
                }
                Inner node = (Inner)active.hashable;
                TreeRange left = new TreeRange(this.tree, (Token)active.left, node.token, MerkleTree.inc(active.depth), node.lchild);
                TreeRange right = new TreeRange(this.tree, node.token, (Token)active.right, MerkleTree.inc(active.depth), node.rchild);
                if (right.isWrapAround()) {
                    this.tovisit.addLast(left);
                    this.tovisit.addFirst(right);
                    continue;
                }
                this.tovisit.addFirst(right);
                this.tovisit.addFirst(left);
            }
            return (TreeRange)this.endOfData();
        }

        @Override
        public Iterator<TreeRange> iterator() {
            return this;
        }
    }

    public static class TreeRange
    extends Range<Token> {
        public static final long serialVersionUID = 1L;
        private final MerkleTree tree;
        public final byte depth;
        private final Hashable hashable;

        TreeRange(MerkleTree tree, Token left, Token right, byte depth, Hashable hashable) {
            super(left, right);
            this.tree = tree;
            this.depth = depth;
            this.hashable = hashable;
        }

        public void hash(byte[] hash) {
            assert (this.tree != null) : "Not intended for modification!";
            this.hashable.hash(hash);
        }

        public byte[] hash() {
            return this.hashable.hash();
        }

        public void addHash(RowHash entry) {
            assert (this.tree != null) : "Not intended for modification!";
            assert (this.hashable instanceof Leaf);
            this.hashable.addHash(entry.hash, entry.size);
        }

        public void ensureHashInitialised() {
            assert (this.tree != null) : "Not intended for modification!";
            assert (this.hashable instanceof Leaf);
            if (this.hashable.hash == null) {
                this.hashable.hash = EMPTY_HASH;
            }
        }

        public void addAll(Iterator<RowHash> entries) {
            while (entries.hasNext()) {
                this.addHash(entries.next());
            }
        }

        @Override
        public String toString() {
            StringBuilder buff = new StringBuilder("#<TreeRange ");
            buff.append(super.toString()).append(" depth=").append(this.depth);
            return buff.append(">").toString();
        }
    }

    public static class TreeDifference
    extends TreeRange {
        private static final long serialVersionUID = 6363654174549968183L;
        private long sizeOnLeft;
        private long sizeOnRight;
        private long rowsOnLeft;
        private long rowsOnRight;

        void setSize(long sizeOnLeft, long sizeOnRight) {
            this.sizeOnLeft = sizeOnLeft;
            this.sizeOnRight = sizeOnRight;
        }

        void setRows(long rowsOnLeft, long rowsOnRight) {
            this.rowsOnLeft = rowsOnLeft;
            this.rowsOnRight = rowsOnRight;
        }

        public long sizeOnLeft() {
            return this.sizeOnLeft;
        }

        public long sizeOnRight() {
            return this.sizeOnRight;
        }

        public long rowsOnLeft() {
            return this.rowsOnLeft;
        }

        public long rowsOnRight() {
            return this.rowsOnRight;
        }

        public TreeDifference(Token left, Token right, byte depth) {
            super(null, left, right, depth, null);
        }

        public long totalRows() {
            return this.rowsOnLeft + this.rowsOnRight;
        }
    }

    public static class MerkleTreeSerializer
    implements IVersionedSerializer<MerkleTree> {
        @Override
        public void serialize(MerkleTree mt, DataOutputPlus out, int version) throws IOException {
            out.writeByte(mt.hashdepth);
            out.writeLong(mt.maxsize);
            out.writeLong(mt.size);
            out.writeUTF(mt.partitioner.getClass().getCanonicalName());
            Token.serializer.serialize((Token)mt.fullRange.left, out, version);
            Token.serializer.serialize((Token)mt.fullRange.right, out, version);
            Hashable.serializer.serialize(mt.root, out, version);
        }

        @Override
        public MerkleTree deserialize(DataInputPlus in, int version) throws IOException {
            IPartitioner partitioner;
            byte hashdepth = in.readByte();
            long maxsize = in.readLong();
            long size = in.readLong();
            try {
                partitioner = FBUtilities.newPartitioner(in.readUTF());
            }
            catch (ConfigurationException e) {
                throw new IOException(e);
            }
            Token left = Token.serializer.deserialize(in, partitioner, version);
            Token right = Token.serializer.deserialize(in, partitioner, version);
            Range<Token> fullRange = new Range<Token>(left, right);
            MerkleTree mt = new MerkleTree(partitioner, fullRange, hashdepth, maxsize);
            mt.size = size;
            mt.root = (Hashable)Hashable.serializer.deserialize(in, partitioner, version);
            return mt;
        }

        @Override
        public long serializedSize(MerkleTree mt, int version) {
            long size = 1 + TypeSizes.sizeof(mt.maxsize) + TypeSizes.sizeof(mt.size) + TypeSizes.sizeof(mt.partitioner.getClass().getCanonicalName());
            size += Token.serializer.serializedSize((Token)mt.fullRange.left, version);
            size += Token.serializer.serializedSize((Token)mt.fullRange.right, version);
            return size += Hashable.serializer.serializedSize(mt.root, version);
        }
    }
}

