/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.StartupException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StartupCheck;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NativeLibrary;
import org.apache.cassandra.utils.SigarLibrary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StartupChecks {
    private static final Logger logger = LoggerFactory.getLogger(StartupChecks.class);
    private final List<StartupCheck> preFlightChecks = new ArrayList<StartupCheck>();
    private final List<StartupCheck> DEFAULT_TESTS = ImmutableList.of((Object)checkJemalloc, (Object)checkValidLaunchDate, (Object)checkJMXPorts, (Object)checkJMXProperties, (Object)inspectJvmOptions, (Object)checkNativeLibraryInitialization, (Object)initSigarLibrary, (Object)checkMaxMapCount, (Object)checkDataDirs, (Object)checkSSTablesFormat, (Object)checkSystemKeyspaceState, (Object)checkDatacenter, (Object[])new StartupCheck[]{checkRack});
    public static final StartupCheck checkJemalloc = new StartupCheck(){

        @Override
        public void execute() {
            if (FBUtilities.isWindows) {
                return;
            }
            String jemalloc = System.getProperty("cassandra.libjemalloc");
            if (jemalloc == null) {
                logger.warn("jemalloc shared library could not be preloaded to speed up memory allocations");
            } else if ("-".equals(jemalloc)) {
                logger.info("jemalloc preload explicitly disabled");
            } else {
                logger.info("jemalloc seems to be preloaded from {}", (Object)jemalloc);
            }
        }
    };
    public static final StartupCheck checkValidLaunchDate = new StartupCheck(){
        private static final long EARLIEST_LAUNCH_DATE = 1215820800000L;

        @Override
        public void execute() throws StartupException {
            long now = System.currentTimeMillis();
            if (now < 1215820800000L) {
                throw new StartupException(1, String.format("current machine time is %s, but that is seemingly incorrect. exiting now.", new Date(now).toString()));
            }
        }
    };
    public static final StartupCheck checkJMXPorts = new StartupCheck(){

        @Override
        public void execute() {
            String jmxPort = System.getProperty("cassandra.jmx.remote.port");
            if (jmxPort == null) {
                logger.warn("JMX is not enabled to receive remote connections. Please see cassandra-env.sh for more info.");
                jmxPort = System.getProperty("cassandra.jmx.local.port");
                if (jmxPort == null) {
                    logger.error("cassandra.jmx.local.port missing from cassandra-env.sh, unable to start local JMX service.");
                }
            } else {
                logger.info("JMX is enabled to receive remote connections on port: {}", (Object)jmxPort);
            }
        }
    };
    public static final StartupCheck checkJMXProperties = new StartupCheck(){

        @Override
        public void execute() {
            if (System.getProperty("com.sun.management.jmxremote.port") != null) {
                logger.warn("Use of com.sun.management.jmxremote.port at startup is deprecated. Please use cassandra.jmx.remote.port instead.");
            }
        }
    };
    public static final StartupCheck inspectJvmOptions = new StartupCheck(){

        @Override
        public void execute() {
            String javaVmName;
            if (!DatabaseDescriptor.hasLargeAddressSpace()) {
                logger.warn("32bit JVM detected.  It is recommended to run Cassandra on a 64bit JVM for better performance.");
            }
            if ((javaVmName = System.getProperty("java.vm.name")).contains("OpenJDK")) {
                logger.warn("OpenJDK is not recommended. Please upgrade to the newest Oracle Java release");
            } else if (!javaVmName.contains("HotSpot")) {
                logger.warn("Non-Oracle JVM detected.  Some features, such as immediate unmap of compacted SSTables, may not work as intended");
            }
        }
    };
    public static final StartupCheck checkNativeLibraryInitialization = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            if (!NativeLibrary.isAvailable()) {
                throw new StartupException(1, "The native library could not be initialized properly. ");
            }
        }
    };
    public static final StartupCheck initSigarLibrary = new StartupCheck(){

        @Override
        public void execute() {
            SigarLibrary.instance.warnIfRunningInDegradedMode();
        }
    };
    public static final StartupCheck checkMaxMapCount = new StartupCheck(){
        private final long EXPECTED_MAX_MAP_COUNT = 1048575L;
        private final String MAX_MAP_COUNT_PATH = "/proc/sys/vm/max_map_count";

        /*
         * Enabled aggressive exception aggregation
         */
        private long getMaxMapCount() {
            block19: {
                Path path = Paths.get("/proc/sys/vm/max_map_count", new String[0]);
                try {
                    Throwable throwable = null;
                    try (BufferedReader bufferedReader = Files.newBufferedReader(path);){
                        String data = bufferedReader.readLine();
                        if (data == null) break block19;
                        try {
                            long l = Long.parseLong(data);
                            return l;
                        }
                        catch (NumberFormatException e) {
                            try {
                                logger.warn("Unable to parse {}.", (Object)path, (Object)e);
                            }
                            catch (Throwable throwable2) {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            catch (Throwable throwable3) {
                                throw throwable3;
                            }
                        }
                    }
                }
                catch (IOException e) {
                    logger.warn("IO exception while reading file {}.", (Object)path, (Object)e);
                }
            }
            return -1L;
        }

        @Override
        public void execute() {
            if (!FBUtilities.isLinux) {
                return;
            }
            if (DatabaseDescriptor.getDiskAccessMode() == Config.DiskAccessMode.standard && DatabaseDescriptor.getIndexAccessMode() == Config.DiskAccessMode.standard) {
                return;
            }
            long maxMapCount = this.getMaxMapCount();
            if (maxMapCount < 1048575L) {
                logger.warn("Maximum number of memory map areas per process (vm.max_map_count) {} is too low, recommended value: {}, you can change it with sysctl.", (Object)maxMapCount, (Object)1048575L);
            }
        }
    };
    public static final StartupCheck checkDataDirs = () -> {
        Iterable dirs = Iterables.concat(Arrays.asList(DatabaseDescriptor.getAllDataFileLocations()), Arrays.asList(DatabaseDescriptor.getCommitLogLocation(), DatabaseDescriptor.getSavedCachesLocation(), DatabaseDescriptor.getHintsDirectory().getAbsolutePath()));
        for (String dataDir : dirs) {
            logger.debug("Checking directory {}", (Object)dataDir);
            File dir = new File(dataDir);
            if (!dir.exists()) {
                logger.warn("Directory {} doesn't exist", (Object)dataDir);
                if (!dir.mkdirs()) {
                    throw new StartupException(3, "Has no permission to create directory " + dataDir);
                }
            }
            if (Directories.verifyFullPermissions(dir, dataDir)) continue;
            throw new StartupException(3, "Insufficient permissions on directory " + dataDir);
        }
    };
    public static final StartupCheck checkSSTablesFormat = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            final HashSet invalid = new HashSet();
            final HashSet<String> nonSSTablePaths = new HashSet<String>();
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getCommitLogLocation()));
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getSavedCachesLocation()));
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getHintsDirectory()));
            SimpleFileVisitor<Path> sstableVisitor = new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    if (!Descriptor.isValidFile(file.getFileName().toString())) {
                        return FileVisitResult.CONTINUE;
                    }
                    try {
                        if (!Descriptor.fromFilename(file.toString()).isCompatible()) {
                            invalid.add(file.toString());
                        }
                    }
                    catch (Exception e) {
                        invalid.add(file.toString());
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    String name = dir.getFileName().toString();
                    return name.equals("snapshots") || name.equals("backups") || nonSSTablePaths.contains(dir.toFile().getCanonicalPath()) ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
                }
            };
            for (String dataDir : DatabaseDescriptor.getAllDataFileLocations()) {
                try {
                    Files.walkFileTree(Paths.get(dataDir, new String[0]), (FileVisitor<? super Path>)sstableVisitor);
                }
                catch (IOException e) {
                    throw new StartupException(3, "Unable to verify sstable files on disk", e);
                }
            }
            if (!invalid.isEmpty()) {
                throw new StartupException(3, String.format("Detected unreadable sstables %s, please check NEWS.txt and ensure that you have upgraded through all required intermediate versions, running upgradesstables", Joiner.on((String)",").join(invalid)));
            }
        }
    };
    public static final StartupCheck checkSystemKeyspaceState = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            for (CFMetaData cfm : Schema.instance.getTablesAndViews("system")) {
                ColumnFamilyStore.scrubDataDirectories(cfm);
            }
            try {
                SystemKeyspace.checkHealth();
            }
            catch (ConfigurationException e) {
                throw new StartupException(100, "Fatal exception during initialization", e);
            }
        }
    };
    public static final StartupCheck checkDatacenter = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            String currentDc;
            String storedDc;
            if (!Boolean.getBoolean("cassandra.ignore_dc") && (storedDc = SystemKeyspace.getDatacenter()) != null && !storedDc.equals(currentDc = DatabaseDescriptor.getEndpointSnitch().getDatacenter(FBUtilities.getBroadcastAddress()))) {
                String formatMessage = "Cannot start node if snitch's data center (%s) differs from previous data center (%s). Please fix the snitch configuration, decommission and rebootstrap this node or use the flag -Dcassandra.ignore_dc=true.";
                throw new StartupException(100, String.format(formatMessage, currentDc, storedDc));
            }
        }
    };
    public static final StartupCheck checkRack = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            String currentRack;
            String storedRack;
            if (!Boolean.getBoolean("cassandra.ignore_rack") && (storedRack = SystemKeyspace.getRack()) != null && !storedRack.equals(currentRack = DatabaseDescriptor.getEndpointSnitch().getRack(FBUtilities.getBroadcastAddress()))) {
                String formatMessage = "Cannot start node if snitch's rack (%s) differs from previous rack (%s). Please fix the snitch configuration, decommission and rebootstrap this node or use the flag -Dcassandra.ignore_rack=true.";
                throw new StartupException(100, String.format(formatMessage, currentRack, storedRack));
            }
        }
    };

    public StartupChecks withDefaultTests() {
        this.preFlightChecks.addAll(this.DEFAULT_TESTS);
        return this;
    }

    public StartupChecks withTest(StartupCheck test) {
        this.preFlightChecks.add(test);
        return this;
    }

    public void verify() throws StartupException {
        for (StartupCheck test : this.preFlightChecks) {
            test.execute();
        }
    }
}

