/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.r2dbc.connectionfactory;

import io.r2dbc.spi.Connection;
import io.r2dbc.spi.ConnectionFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.reactivestreams.Publisher;
import org.springframework.core.Ordered;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.data.r2dbc.connectionfactory.ConnectionHolder;
import org.springframework.data.r2dbc.connectionfactory.ConnectionProxy;
import org.springframework.data.r2dbc.connectionfactory.DelegatingConnectionFactory;
import org.springframework.data.r2dbc.connectionfactory.ReactiveTransactionSynchronization;
import org.springframework.data.r2dbc.connectionfactory.SingletonConnectionFactory;
import org.springframework.data.r2dbc.connectionfactory.TransactionResources;
import org.springframework.lang.Nullable;
import org.springframework.transaction.NoTransactionException;
import org.springframework.transaction.reactive.TransactionSynchronization;
import org.springframework.transaction.reactive.TransactionSynchronizationManager;
import org.springframework.util.Assert;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

public abstract class ConnectionFactoryUtils {
    public static final int CONNECTION_SYNCHRONIZATION_ORDER = 1000;
    private static final Log logger = LogFactory.getLog(ConnectionFactoryUtils.class);

    private ConnectionFactoryUtils() {
    }

    public static Mono<Tuple2<Connection, ConnectionFactory>> getConnection(ConnectionFactory connectionFactory) {
        return ConnectionFactoryUtils.doGetConnection(connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to obtain R2DBC Connection", e));
    }

    public static Mono<Tuple2<Connection, ConnectionFactory>> doGetConnection(ConnectionFactory connectionFactory) {
        Assert.notNull((Object)connectionFactory, (String)"ConnectionFactory must not be null!");
        return TransactionSynchronizationManager.currentTransaction().flatMap(synchronizationManager -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)synchronizationManager.getResource((Object)connectionFactory)));
            if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
                conHolder.requested();
                if (!conHolder.hasConnection()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug((Object)"Fetching resumed R2DBC Connection from ConnectionFactory");
                    }
                    return ConnectionFactoryUtils.fetchConnection(connectionFactory).doOnNext(conHolder::setConnection);
                }
                return Mono.just((Object)conHolder.getConnection());
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Fetching R2DBC Connection from ConnectionFactory");
            }
            Mono<Connection> con = ConnectionFactoryUtils.fetchConnection(connectionFactory);
            if (synchronizationManager.isSynchronizationActive()) {
                return con.flatMap(it -> Mono.just((Object)it).doOnNext(conn -> {
                    ConnectionHolder holderToUse = conHolder;
                    if (holderToUse == null) {
                        holderToUse = new ConnectionHolder((Connection)conn);
                    } else {
                        holderToUse.setConnection((Connection)conn);
                    }
                    holderToUse.requested();
                    synchronizationManager.registerSynchronization((TransactionSynchronization)new ConnectionSynchronization(holderToUse, connectionFactory));
                    holderToUse.setSynchronizedWithTransaction(true);
                    if (holderToUse != conHolder) {
                        synchronizationManager.bindResource((Object)connectionFactory, (Object)holderToUse);
                    }
                }).onErrorResume(e -> ConnectionFactoryUtils.releaseConnection(it, connectionFactory).then(Mono.error((Throwable)e))));
            }
            return con;
        }).map(conn -> Tuples.of((Object)conn, (Object)connectionFactory)).onErrorResume(NoTransactionException.class, e -> Mono.subscriberContext().flatMap(it -> {
            if (it.hasKey(ReactiveTransactionSynchronization.class)) {
                ReactiveTransactionSynchronization synchronization = (ReactiveTransactionSynchronization)it.get(ReactiveTransactionSynchronization.class);
                return ConnectionFactoryUtils.obtainConnection(synchronization, connectionFactory);
            }
            return Mono.empty();
        }).switchIfEmpty(Mono.defer(() -> Mono.from((Publisher)connectionFactory.create()).map(it -> Tuples.of((Object)it, (Object)connectionFactory)))));
    }

    private static Mono<Tuple2<Connection, ConnectionFactory>> obtainConnection(ReactiveTransactionSynchronization synchronization, ConnectionFactory connectionFactory) {
        if (synchronization.isSynchronizationActive()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Registering transaction synchronization for R2DBC Connection");
            }
            TransactionResources txContext = synchronization.getCurrentTransaction();
            ConnectionFactory resource = txContext.getResource(ConnectionFactory.class);
            Mono attachNewConnection = Mono.defer(() -> Mono.from((Publisher)connectionFactory.create()).map(it -> {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)"Fetching new R2DBC Connection from ConnectionFactory");
                }
                SingletonConnectionFactory s = new SingletonConnectionFactory(connectionFactory.getMetadata(), (Connection)it);
                txContext.registerResource(ConnectionFactory.class, s);
                return Tuples.of((Object)it, (Object)connectionFactory);
            }));
            return Mono.justOrEmpty((Object)resource).flatMap(ConnectionFactoryUtils::createConnection).switchIfEmpty(attachNewConnection);
        }
        return Mono.empty();
    }

    private static Mono<Connection> fetchConnection(ConnectionFactory connectionFactory) {
        Publisher con = connectionFactory.create();
        if (con == null) {
            throw new IllegalStateException("ConnectionFactory returned null from getConnection(): " + connectionFactory);
        }
        return Mono.from((Publisher)con);
    }

    public static Mono<Void> releaseConnection(@Nullable Connection con, @Nullable ConnectionFactory connectionFactory) {
        return ConnectionFactoryUtils.doReleaseConnection(con, connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to close R2DBC Connection", e));
    }

    public static Mono<Void> doReleaseConnection(@Nullable Connection con, @Nullable ConnectionFactory connectionFactory) {
        return TransactionSynchronizationManager.currentTransaction().flatMap(it -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)it.getResource((Object)connectionFactory)));
            if (conHolder != null && ConnectionFactoryUtils.connectionEquals(conHolder, con)) {
                conHolder.released();
            }
            return Mono.from((Publisher)con.close());
        }).onErrorResume(NoTransactionException.class, e -> {
            if (connectionFactory instanceof SingletonConnectionFactory) {
                SingletonConnectionFactory factory = (SingletonConnectionFactory)connectionFactory;
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)"Releasing R2DBC Connection");
                }
                return factory.close(con);
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Closing R2DBC Connection");
            }
            return Mono.from((Publisher)con.close());
        });
    }

    public static Mono<Void> closeConnection(Connection connection, ConnectionFactory connectionFactory) {
        return ConnectionFactoryUtils.doCloseConnection(connection, connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to obtain R2DBC Connection", e));
    }

    public static Mono<Void> doCloseConnection(Connection connection, @Nullable ConnectionFactory connectionFactory) {
        if (!(connectionFactory instanceof SingletonConnectionFactory) || ((SingletonConnectionFactory)connectionFactory).shouldClose(connection)) {
            SingletonConnectionFactory factory = (SingletonConnectionFactory)connectionFactory;
            return factory.close(connection).then(Mono.from((Publisher)connection.close()));
        }
        return Mono.empty();
    }

    public static Mono<ReactiveTransactionSynchronization> currentReactiveTransactionSynchronization() {
        return Mono.subscriberContext().filter(it -> it.hasKey(ReactiveTransactionSynchronization.class)).switchIfEmpty(Mono.error((Throwable)new NoTransactionException("Transaction management is not enabled. Make sure to register ReactiveTransactionSynchronization in the subscriber Context!"))).map(it -> (ReactiveTransactionSynchronization)it.get(ReactiveTransactionSynchronization.class));
    }

    public static Mono<ReactiveTransactionSynchronization> currentActiveReactiveTransactionSynchronization() {
        return ConnectionFactoryUtils.currentReactiveTransactionSynchronization().filter(ReactiveTransactionSynchronization::isSynchronizationActive).switchIfEmpty(Mono.error((Throwable)new NoTransactionException("ReactiveTransactionSynchronization not active!")));
    }

    public static Mono<ConnectionFactory> currentConnectionFactory(ConnectionFactory connectionFactory) {
        return TransactionSynchronizationManager.currentTransaction().filter(TransactionSynchronizationManager::isSynchronizationActive).filter(it -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)it.getResource((Object)connectionFactory)));
            return conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction());
        }).map(it -> connectionFactory).onErrorResume(NoTransactionException.class, ConnectionFactoryUtils::obtainDefaultConnectionFactory);
    }

    private static boolean connectionEquals(ConnectionHolder conHolder, Connection passedInCon) {
        if (!conHolder.hasConnection()) {
            return false;
        }
        Connection heldCon = conHolder.getConnection();
        return heldCon == passedInCon || heldCon.equals(passedInCon) || ConnectionFactoryUtils.getTargetConnection(heldCon).equals(passedInCon);
    }

    public static Connection getTargetConnection(Connection con) {
        Connection conToUse = con;
        while (conToUse instanceof ConnectionProxy) {
            conToUse = ((ConnectionProxy)conToUse).getTargetConnection();
        }
        return conToUse;
    }

    private static int getConnectionSynchronizationOrder(ConnectionFactory connectionFactory) {
        int order = 1000;
        ConnectionFactory current = connectionFactory;
        while (current instanceof DelegatingConnectionFactory) {
            --order;
            current = ((DelegatingConnectionFactory)current).getTargetConnectionFactory();
        }
        return order;
    }

    private static Mono<? extends ConnectionFactory> obtainDefaultConnectionFactory(NoTransactionException e) {
        return ConnectionFactoryUtils.currentActiveReactiveTransactionSynchronization().map(synchronization -> {
            TransactionResources currentSynchronization = synchronization.getCurrentTransaction();
            return currentSynchronization.getResource(ConnectionFactory.class);
        }).switchIfEmpty(Mono.error((Throwable)new DataAccessResourceFailureException("Cannot extract ConnectionFactory from current TransactionContext!")));
    }

    private static Mono<Tuple2<Connection, ConnectionFactory>> createConnection(ConnectionFactory factory) {
        if (logger.isDebugEnabled()) {
            logger.debug((Object)"Fetching resumed R2DBC Connection from ConnectionFactory");
        }
        return Mono.from((Publisher)factory.create()).map(connection -> Tuples.of((Object)connection, (Object)factory));
    }

    private static class ConnectionSynchronization
    implements TransactionSynchronization,
    Ordered {
        private final ConnectionHolder connectionHolder;
        private final ConnectionFactory connectionFactory;
        private int order;
        private boolean holderActive = true;

        ConnectionSynchronization(ConnectionHolder connectionHolder, ConnectionFactory connectionFactory) {
            this.connectionHolder = connectionHolder;
            this.connectionFactory = connectionFactory;
            this.order = ConnectionFactoryUtils.getConnectionSynchronizationOrder(connectionFactory);
        }

        public int getOrder() {
            return this.order;
        }

        public Mono<Void> suspend() {
            if (this.holderActive) {
                return TransactionSynchronizationManager.currentTransaction().flatMap(it -> {
                    it.unbindResource((Object)this.connectionFactory);
                    if (this.connectionHolder.hasConnection() && !this.connectionHolder.isOpen()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory).doOnTerminate(() -> this.connectionHolder.setConnection(null));
                    }
                    return Mono.empty();
                });
            }
            return Mono.empty();
        }

        public Mono<Void> resume() {
            if (this.holderActive) {
                return TransactionSynchronizationManager.currentTransaction().doOnNext(it -> it.bindResource((Object)this.connectionFactory, (Object)this.connectionHolder)).then();
            }
            return Mono.empty();
        }

        public Mono<Void> beforeCompletion() {
            if (!this.connectionHolder.isOpen()) {
                return TransactionSynchronizationManager.currentTransaction().flatMap(it -> {
                    it.unbindResource((Object)this.connectionFactory);
                    this.holderActive = false;
                    if (this.connectionHolder.hasConnection()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory);
                    }
                    return Mono.empty();
                });
            }
            return Mono.empty();
        }

        public Mono<Void> afterCompletion(int status) {
            if (this.holderActive) {
                return TransactionSynchronizationManager.currentTransaction().flatMap(it -> {
                    it.unbindResourceIfPossible((Object)this.connectionFactory);
                    this.holderActive = false;
                    if (this.connectionHolder.hasConnection()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory).doOnTerminate(() -> this.connectionHolder.setConnection(null));
                    }
                    return Mono.empty();
                });
            }
            this.connectionHolder.reset();
            return Mono.empty();
        }
    }
}

